<?php
###############################################################################
# Settings.php
#
# @author Anil Kumar <akumar@codepunch.com>
# @link   https://codepunch.com
#
############################################################################### 

namespace 	CodePunch\Config;
use CodePunch\Base\Util as UTIL;

###############################################################################

class Settings  {
	
	private $authentication = null;
	
	###########################################################################
	
	public function __construct($auth)
	{
		$this->authentication = $auth;
	}
	
	###########################################################################
	
	public function getServerID()
	{
		return $this->authentication->getServerID();
	}
	
	###########################################################################
	
	public function getOption($name, $defvalue, $server="*", $forceintotable=false, $mode=0)
	{
		$value = $defvalue;
		$colname = ($mode == 0 ? "svalue" : "tvalue");
		if($this->authentication) {
			$db = $this->authentication->getDatabase();
			if($db) {
				$serverid = ($server == null) ? $this->getServerID() : $server;
				$rows = $db->getFromTable($colname, $db->getSettingsTableName(), "name=? AND server=?", array($name, $serverid));
				if($rows === false  || !isset($rows[0])) {
					if($forceintotable)
						$db->insertIntoTable($db->getSettingsTableName(), array('name'=>$name, $colname=>$defvalue, 'server'=>$serverid));
				}
				else {
					$row = $db->fetchRow($rows, 0);
					$value = $row[$colname];
				}
			}
		}
		return $value;
	}
	
	###########################################################################
	
	public function getString($name, $defvalue, $server="*", $forceintotable=false) {
		return $this->getOption($name, $defvalue, $server, $forceintotable, 0);
	}
	
	###########################################################################
	
	public function getText($name, $defvalue, $server="*", $forceintotable=false) {
		return $this->getOption($name, $defvalue, $server, $forceintotable, 1);
	}
	
	###########################################################################
	
	public function getEncryptedOption($name, $defvalue, $server="*", $forceintotable=false)
	{
		$db = $this->authentication->getDatabase();
		$serverid = ($server == null) ? $this->getServerID() : $server;
		$rows = $db->getFromTable('svalue', $db->getSettingsTableName(), "name=? AND server=?", array($name, $serverid));
		if($rows === false  || !isset($rows[0])) {
			$row = $defvalue;
			if($forceintotable) {
				$encrypted = $this->authentication->encrypt($row);
				$db->insertIntoTable($db->getSettingsTableName(), array('name'=>$name, 'svalue'=>$encrypted, 'server'=>$serverid));
			}
		}
		else {
			$row = $db->fetchRow($rows, 0);
			$row = $row['svalue'];
			if($row != "")
				$row = $this->authentication->decrypt($row);
		}
		return $row;
	}
	
	###########################################################################
	
	public function getBoolean($name, $defvalue=false, $server="*", $forceintotable=false)
	{
		$strbool = $defvalue ? "true" : "false";
		$value = $this->getOption($name, $strbool, $server, $forceintotable);
		return UTIL::str_to_bool($value);
	}
	
	###########################################################################
	
	public function getPendingJobs()
	{
		$pendingjobs = false;
		$fp = UTIL::get_lock_file("jobs");
		if($fp !== false) {
			$jobs = $this->getText("jobs", "", "*");
			if($jobs != "") {
				$this->setText("jobs", "", "*");
				$pendingjobs = explode("\n", $jobs);
			}
			flock($fp, LOCK_UN);
			fclose($fp);
		}
		return $pendingjobs;
	}
	
	###########################################################################
	
	public function addPendingJob($job)
	{
		$status = false;
		$fp = UTIL::get_lock_file("jobs");
		if($fp !== false) {
			$jobs = $this->getText("jobs", "", "*");
			if($jobs != "") 
				$jobs = trim($jobs) . "\n" . $job;
			else
				$jobs = trim($job);
			$status = $this->setText("jobs", $jobs, "*");
			flock($fp, LOCK_UN);
			fclose($fp);
		}
		return $status;
	}
	
	###########################################################################
	
	public function getPendingAlerts($interval, $alertname)
	{
		$alertdata = false;
		$fp = UTIL::get_lock_file("alerts");
		if($fp !== false) {
			$optioname = "pending_" . strtolower($alertname) . "_alerts";
			$emailname = strtolower($alertname) . "_alert_recipients";
			$lastinsertedat = $this->getOption($optioname, date("Y-m-d H:i:s"));
			if($lastinsertedat != "") {
				$removeat = strtotime($lastinsertedat) + ($interval*60);
				if(time() >= $removeat) {
					$msg = $this->getText($optioname, "", "*");
					if($msg != "") {
						$msg = explode("\n", $msg);
						$alertdata = $this->getDNSAlert($msg, $alertname);
					}
					$this->setText($optioname, "", "*");
					$this->setOption($optioname, "");
				}
			}

			if($alertdata === false) {
				$alert_on_app_update = $this->getBoolean("alert_on_app_update", false);
				if($alert_on_app_update) {
					$last_checked_at = $this->getOption("last_app_update_check", date("Y-m-d H:i:s", time()-21601));
					$nextat = strtotime($last_checked_at) + 21600;
					if(time() >= $nextat) {
						$alertdata = $this->getAppUpdateAlert();
						$this->setOption("last_app_update_check", date("Y-m-d H:i:s"));
					}
				}
			}

			flock($fp, LOCK_UN);
			fclose($fp);
		}
		return $alertdata;
	}

	###########################################################################

	public function addPendingAlerts($emailtxt, $alertname)
	{
		$status = false;
		$fp = UTIL::get_lock_file("alerts");
		if($fp !== false) {
			$optioname = "pending_" . strtolower($alertname) . "_alerts";
			$db = $this->authentication->getDatabase();
			$message = $this->getText($optioname, "", "*");
			if($message != "")
				$message = trim($message) . "\n" . $emailtxt;
			else
				$message = trim($emailtxt);
			$thistime = date("Y-m-d H:i:s");
			$serverid = "*";
			$row = $db->hasRow($db->getSettingsTableName(), array('name','server'), array($optioname, $serverid));
			if($row === false)
				$status = $db->insertIntoTable($db->getSettingsTableName(), array('name'=>$optioname, 'svalue'=>$thistime, 'tvalue'=>$message, 'server'=>$serverid));
			else {
				$setdata = array('svalue'=>$thistime, 'tvalue'=>$message);
				$rows = $db->getFromTable("svalue", $db->getSettingsTableName(), "name=? AND server=?", array($optioname, $serverid));
				if($rows !== false && count($rows)) {
					$row = $db->fetchRow($rows, 0);
					if($row['svalue'] != "" && strlen($row['svalue']) > 4)
						$setdata = array('tvalue'=>$message);
				}
				$status = $db->updateTable($db->getSettingsTableName(), $setdata, "name=? AND server=?", array($optioname, $serverid));
			}
			flock($fp, LOCK_UN);
			fclose($fp);
		}
		return $status;
	}

	###########################################################################

	public function getDNSAlert($alerts, $alertname)
	{
		foreach($alerts as &$m) {
			if(stristr($m, " deleted ") !== false)
				$m = "<li class=\"deleted\">$m</li>";
			else
				$m = "<li class=\"found\">$m</li>";
		}
		$msg = "<ol class=\"alerts\">" . implode("\n", $alerts) . "</ol>";
		$rooturl = $this->getOption("application_root_url", "");
		$product = \CodePunch\Config\Defaults::product();
		$alertdata = array();
		$alertdata['subject'] = str_replace("Server Edition", "SED", "$product Alert");
		$alertdata['body'] = $this->formatUpdateAlert("$alertname Alerts", $msg, $rooturl, $product);
		$alertdata['emailname'] = strtolower($alertname) . "_alert_recipients";
		return $alertdata;
	}

	###########################################################################

	public function getAppUpdateAlert()
	{
		$alertdata = false;
		$ai = $this->authentication->getApplicationStatus();
		$avbuild = $ai['newversion'];
		$thisbuild = $ai['version'];
		$this->setOption("available_app_build_date", $avbuild);
		$this->setOption("installed_app_build_date", $thisbuild);
		$this->setOption("app_update_available", $ai['update']);
		if($ai['update'] == 1) {
			$app_update_alert_email = $this->getOption("app_update_alert_email", "");
			if($app_update_alert_email != "") {
				$rooturl = $this->getOption("application_root_url", "");

				$body  = "<p>A new version of <b>" . $ai['product'] . "</b> is available for download and installation.</p>";
				$body .= "<p>Currently Installed Version &amp; Build Date: $thisbuild</p>";
				$body .= "<p>Available Version &amp; Build Date: <b>$avbuild</b></p>";
				$body .= "<p>Download from <a href=\"" . $ai['download'] . "\">here</a> or use the command line tool to automatically download and install. For more details, please see the instructions at the link below.</p>";
				$body .= "<p><a class=\"update-button\" href=\"https://domainpunch.com/sed/guidev5/kb/updates.php\">How to update</a></p>";

				$alertdata['body'] = $this->formatUpdateAlert("An Update is Available", $body, $rooturl, $ai['product']);
				$alertdata['subject'] = $ai['product'] . " Update";
				$alertdata['emailname'] = "app_update_alert_email";
			}
		}
		UTIL::debug_cli_print($ai);
		return $alertdata;
	}

	###########################################################################

	public function formatUpdateAlert($subject, $msgbody, $rooturl, $product)
	{
		$folder = UTIL::get_install_folder_path() . "lib/layouts/modules/reports/";
		$filename = $folder . "emails.alert.htm";
		if(!is_file($filename))
			$filename = $folder . "emails.alert.custom.htm";
		if(is_file($filename)) {
			$alerttext = file_get_contents($filename);
			$alerttext = str_ireplace("{{SUBJECT}}", $subject, $alerttext);
			$alerttext = str_ireplace("{{APPROOTURL}}", $rooturl, $alerttext);
			$alerttext = str_ireplace("{{PRODUCT}}", $product, $alerttext);
			$alerttext = str_ireplace("{{ALERTDATA}}", $msgbody, $alerttext);
			
			$platform = $this->authentication->getDatabase()->connection->getDatabasePlatform();
			$driver = $platform->getName();
			$alerttext = str_ireplace("{{DBDRIVER}}", $driver, $alerttext);
		}
		else
			$alerttext = $msgbody;
		return $alerttext;
	}

	###########################################################################

	public function clearOption($name, $server="*")
	{
		$serverid = ($server == null) ? $this->getServerID() : $server;
		return $this->setOption($name, null, $serverid);
	}
	
	###########################################################################
	
	public function setOption($name, $value, $server="*", $mode=0)
	{
		$colname = ($mode == 0 ? "svalue" : "tvalue");
		if($this->authentication) {
			$db = $this->authentication->getDatabase();
			if($db) {
				$serverid = ($server == null) ? $this->getServerID() : $server;
				$row = $db->hasRow($db->getSettingsTableName(), array('name','server'), array($name, $serverid));
				if($row === false)
					return $db->insertIntoTable($db->getSettingsTableName(), array('name'=>$name, $colname=>$value, 'server'=>$serverid));
				else
					return $db->updateTable($db->getSettingsTableName(), array($colname=>$value), "name=? AND server=?", array($name, $serverid));
			}
		}
		return false;
	}
	
	###########################################################################
	
	public function setText($name, $tvalue, $server="*") {
		return $this->setOption($name, $tvalue, $server="*", 1);
	}
	
	###########################################################################
	
	public function setString($name, $tvalue, $server="*") {
		return $this->setOption($name, $tvalue, $server="*", 0);
	}
	
	###########################################################################
	
	public function setEncryptedOption($name, $value, $server="*")
	{
		$encvalue = $this->authentication->encrypt($value);
		return $this->setOption($name, $encvalue, $server);
	}
	
	###########################################################################
	
	public function getReportColumnModelByName($cmname) 
	{
		$db = $this->authentication->getDatabase();
		$columnlist = $db->findOneOf($db->getReportSchedulerTableName(), "name", $cmname, "columns");
		if($columnlist == "" || $columnlist === false) {
			$clist = self::get_default_columns_for_groups();
			$columnlist = UTIL::get_from_array($clist[$cmname], "");
		}
		$columns = explode(";", $columnlist);
		$columns = array_diff($columns, array('s.r_h_disp', 'r_h_disp'));
		$columnlist = implode(";", array_values($columns));
		if($columnlist == "")
			$columnlist = "hostname;s.ip;s.record_type;ptr;s.added_on;domain";
		
		//return $this->getDomainColumnModel("s.r_h_disp;" . $columnlist);
		// Removed s.r_h_disp from report column model on Jan 2 2022
		// This screws up the row count stats and we don't use the column anyway.
		return $this->getDomainColumnModel($columnlist);
	}
	
	###########################################################################
	
	public function setReportColumnModelByName($cmname, $columnmodel) 
	{
		if(is_array($columnmodel))
			$columnmodel = implode(";", array_values($columnlist));
		$db = $this->authentication->getDatabase();
		if(!$db->hasRow($db->getReportSchedulerTableName(), "name", $cmname)) 
			return $db->insertIntoTable($db->getReportSchedulerTableName(), array('columns'=>$columnmodel,'name'=>$cmname));
		else
			return $db->updateTable($db->getReportSchedulerTableName(), array('columns'=>$columnmodel), "name=?", array($cmname));
	}
	
	###########################################################################
	
	public function getDomainColumnModelByName($cmname) 
	{
		$columnlist = $this->getOption($cmname, "");
		if($columnlist == "") {
			$clist = self::get_default_columns_for_groups();
			$columnlist = UTIL::get_from_array($clist[$cmname], "");
		}
		$columns = explode(";", $columnlist);
		$columns = array_diff($columns, array('r_h_disp'));
		$columns = array_diff($columns, array('domain'));
		$columnlist = implode(";", array_values($columns));
		return $this->getDomainColumnModel("r_h_disp;domain;" . $columnlist, false);
	}
	
	###########################################################################
	
	public function getSubdomainColumnModelByname($cmname) 
	{
		$columnlist = $this->getOption($cmname, "");
		if($columnlist == "") {
			$clist = self::get_default_columns_for_groups();
			$columnlist = UTIL::get_from_array($clist[$cmname], "");
		}
		return $this->getSubdomainColumnModel($columnlist);
	}
	
	###############################################################################

	public static function get_default_columns_for_groups()
	{
		return array(
		'custom01_columns_array' 			=> 'ip;notes_c;notes_d;ns3;ns4',
		'custom02_columns_array' 			=> 'ip;notes_a;notes_b;ns1;ns2',
		'custom03_columns_array' 			=> 'primary_whois_checked_at;rootdns_checked_at;home_page_checked_at;ip_whois_at;ping_checked_at;secondary_whois_checked_at;ssl_checked_at;subdomains_checked_at;manual_edited_at',
		'date_columns_array' 				=> 'registry_expiry;registrar_expiry;created_on;last_update;tld;primary_whois_checked_at',
		'http_columns_array' 				=> 'ip;home_page_url;home_page_title;home_page_status;redirect_urls;home_page_checked_at',
		'registrar_columns_array' 			=> 'registrar;status;last_update;primary_whois_checked_at;ascii_domain',
		'network_columns_array'				=> 'ns1;ns2;ns3;ping_time;mx1;mx2;ip',
		'sitepop_columns_array' 			=> 'ping_time;ip;ip_org_name',
		'notes_columns_array' 				=> 'notes_a;notes_b;notes_c;notes_d;ip',
		'ipdata_columns_array' 				=> 'ip;ip_org_name;ip_net_name;ip_asnumber;ip_asname',

		'subdomaingrid_columns_array' 		=> 'subdomain;auto_added;record_type;record_value;ttl;ip;ptr;added_on',
		'sslgrid_columns_array'				=> 'subdomain;ssl_valid_from;ssl_valid_to;ssl_issued_to;ssl_issued_by;ip;subject_alt_name;subject_key_id;notes_a',

		'Subdomain Records'					=> 'hostname;domain;subdomain;s.ip;record_type;record_value;ptr,domain',
		'Domain Expiry Report'				=> 'domain;registry_expiry;registrar_expiry;created_on;last_update;d.ip;primary_whois_checked_at;secondary_whois_checked_at,registry_expiry',
		'SSL Expiry Report'					=> 'hostname;s.ip;ssl_valid_from;ssl_valid_to;ssl_issued_to;ssl_issued_by;ssl_checked_at;subject_alt_name,ssl_valid_to',
		'Valid SSL Report'					=> 'hostname;s.ip;ssl_valid_from;ssl_valid_to;ssl_issued_to;ssl_issued_by;ssl_checked_at;subject_alt_name,ssl_valid_to',
		'Missing Domain Expiry Dates'		=> 'domain;registrar;status;ns1;ns2;ip;primary_whois_checked_at,registry_expiry',
		'Valid Hostnames'					=> 'hostname;subdomain;domain'
		);
	}
	
	###########################################################################
	
	public function getDomainColumnModel($columnlist, $multitable=true)
	{
		$dateformat = $this->getOption("date_display_format", "YYYY-MMM-DD");
		$db = $this->authentication->getDatabase();
		$domain_column_names = $db->getColumnNames($db->getDomainTableName());
		$subdomain_column_names = $db->getColumnNames($db->getSubdomainTableName());
		$parts = explode(";", $columnlist);
		$j = 0;
		$colmodels = array();
		$current_domain_columns = $db->getDetailsOfAllDomainColumns();
		$current_subdomain_columns = $db->getColumnDetails($db->getSubdomainTableName());
		foreach($parts as $part)
		{
			$cname = strtolower(trim($part));
			
			$columntable = 0;
			if(UTIL::starts_with($cname, "s.")) {
				$columntable = 2;
				$cname = substr($cname, 2);
			}
			else if(UTIL::starts_with($cname, "d.")) {
				$columntable = 1;
				$cname = substr($cname, 2);
				if(!in_array($cname, $domain_column_names))
					continue;
			}
			else if(!strcasecmp($cname, "hostname"))
				$columntable = 2;
			else if(!in_array($cname, $domain_column_names) && in_array($cname, $subdomain_column_names))
				$columntable = 2;
			else if(in_array($cname, $domain_column_names))
				$columntable = 1;

			if(!$multitable && $columntable == 2)
				continue;
			
			if($columntable == 2)
				$cinfo = UTIL::get_from_array($current_subdomain_columns[$cname], array());
			else
				$cinfo = UTIL::get_from_array($current_domain_columns[$cname], array());
			
			$columns[0] = $cname; // Label
			$columns[1] = $cname; // Field Name
			$columns[1] = str_replace(" ","_", $columns[1]);

			$columns[2] = UTIL::get_from_array($cinfo['width'], 100);
			if(!strcasecmp($cname, "hostname")) {
				$columns[2] = UTIL::get_from_array($current_domain_columns['domain']['width'], 160) + 30;
				$cinfo['label'] = 'Hostname';
			}
			$columns[0] = UTIL::get_from_array($cinfo['label'], $cname);
			$colmodel = array();
			if($columns[1] != "" && $columntable != 0)
			{
				$classname = strtolower($columns[1]);
				$classname = str_replace("_","-", $classname) . "-cell";
				if($cname == 'r_h_disp')
					$classname .= " tc-icon";
				
				$colmodel['editable'] = UTIL::str_to_bool(UTIL::get_from_array($cinfo['editable'], false));
				$colmodel['search'] = UTIL::str_to_bool(UTIL::get_from_array($cinfo['gridview'], false));
				$colmodel['sortable'] = UTIL::str_to_bool(UTIL::get_from_array($cinfo['gridview'], false));
				$colmodel['hidden'] = UTIL::get_from_array($cinfo['gridview'], false) ? false : true;
				if($columns[1] == "hostname" || $columns[1] == "days_to_expiry") {
					$colmodel['hidden'] = false;
					if($columns[1] != "days_to_expiry")
						$colmodel['sortable'] = true;
				}
				
				if(strtolower($columns[1]) == "domain") {
					$colmodel['editable'] = true;
					$colmodel['editoptions'] = array('readonly'=>true,'size'=>35);
					if(!$multitable)
						$colmodel['move'] = false;
					$colmodel['minwidth'] = 100;
				}
				else if(strtolower($columns[1]) == "r_h_disp")
				{
					$colmodel['search'] = false;
					$colmodel['fixed'] = true;
					$colmodel['formatter'] = 'highlightFormatter';
					$colmodel['move'] = false;
				}
				else if(strtolower($columns[1]) == "page_token_found")
				{
					$colmodel['align'] = 'center';
					$colmodel['formatter'] = 'booleanFormatter';
				}
				else if(strtolower($columns[1]) == "registrar") {
					$colmodel['editoptions'] = array('size'=>35);
					$colmodel['formatter'] = 'registrarFormatter';
				}
				else if(strtolower($columns[1]) == "availability")
				{
					$colmodel['align'] = 'center';
					$colmodel['formatter'] = 'domAvailFormatter';
				}
				else
				{
					$ft = UTIL::get_from_array($cinfo['fieldtype'], "");
					if($ft !== false)
					{
						if(stristr($ft, "datetime") !== false && $columns[1] != "last_update")
						{
							if(in_array($columns[1], $domain_column_names)) {
								$colmodel['sorttype'] = 'date';
								$colmodel['formatter'] = 'date';
								$colmodel['formatoptions'] = array('srcformat'=>'Y-m-d H:i:s','newformat'=>"$dateformat H:i:s");
							}
							else {
								$colmodel['sorttype'] = 'date';
								$colmodel['formatter'] = 'date';
								$colmodel['formatoptions'] = array('srcformat'=>'Y-m-d H:i:s','newformat'=>"$dateformat H:i:s");
							}
						}
						else if(stristr($ft, "date") !== false) {
							$colmodel['sorttype'] = 'date';
							$colmodel['formatter'] = 'date';
							$colmodel['formatoptions'] = array('srcformat'=>'Y-m-d','newformat'=>$dateformat);
						}
						else if(stristr($ft, "string") !== false) {
							$colmodel['editoptions'] = array('size'=>35);
						}
						else {
							$colmodel['editoptions'] = array('size'=>20);
						}
					}
				}
				
				if(!in_array($columns[1], $domain_column_names))
					$columns[1] = "s." . $columns[1];
				if($columntable == 2 && substr($columns[1],1,1) != ".")
					$columns[1] = "s." . $columns[1];
				if($columntable == 1 && substr($columns[1],1,1) != ".")
					$columns[1] = "d." . $columns[1];
				
				$colmodel['classes'] = "dtcell $classname";
				$colmodel['name'] = $columns[1];
				$colmodel['width'] = $columns[2] >= 50 ? $columns[2] : 50;
				$colmodel['label'] = addslashes($columns[0]);
				$colmodels[] = array_reverse($colmodel);
			}
		}
		return $colmodels;
	}
	
	###########################################################################
	
	public function getSubdomainColumnModel($columnlist) 
	{
		$columns = explode(";", $columnlist);
		foreach($columns as &$c) {
			$column = trim($c);
			$cs = strtolower(substr($column, 1, 1));
			if($cs == ".")
				$column = substr($c, 2);
			$c = "s." . $column;
		}
		$columnlist = implode(";", $columns);
		return $this->getDomainColumnModel($columnlist);
	}
	
	###########################################################################
	
	public function getSSLCertSettings($sslsettings=array()) {
		$sslsettings['emails'] = $this->getOption("sslbrowse-emails", "");
		$sslsettings['autorun'] = $this->getBoolean("sslbrowse-autorun", false);
		$sslsettings['frequency'] = $this->getOption("sslbrowse-frequency", "0");
		$sslsettings['runits'] = $this->getOption("sslbrowse-runits", "0");
		$sslsettings['lastrun'] = $this->getOption("sslbrowse-lastrun", "");
		return $sslsettings;
	}
	
	###########################################################################
	
	public static function getDefaultBranding()
	{
		return array(
			'custom_url_home' => "https://domainpunch.com/sed/?sed5",
			'custom_url_guide' => "https://domainpunch.com/sed/guidev5/?sed5",
			'custom_url_contact' => "https://domainpunch.com/support/contact.php?sed5",
			'custom_url_changelog' => "https://domainpunch.com/sed/changelog.php?sed5",
			'custom_url_buy' => "https://domainpunch.com/sed/buy.php?sed5",
			"main_logo_image" => false, 
			"big_logo_image" => false,
			"background_image" => false,
			"background_property" => false,
			"background_size" => false
		);
	}
	
	###########################################################################
	
	public static function getUserOptions()
	{
		return array(
			'date_display_format'=>'YYYY-MMM-DD',
			'grid_search_on_enter' => false,
			'grid_enable_inline_edit' => true,
			'grid_auto_search_delay' => 800,
			'max_download_rows'	=> 5000,
			'default_home_page' => \CodePunch\Config\ConfigRoot::DEFAULT_HOME_PAGE,
			'alert_on_app_update' => false,
			'minimize_transparency_level' => true,
			'app_update_alert_email' => '',
			
			'do_webshots' => false,
			'webshot_domains' => '[IN_Business Domains] OR [IN_Important Domains] OR [IN_Webshot Domains]',
			'webshot_days' => 30,
			'webshot_maxcount' => 1,
			'webshot_nodejsoptions' => '--ac',
			
			'default_lookup_types' => 'dw,ip'
		);
	}
	
	###########################################################################
	
}
