let authSessionTimer = null;
let loginPollTimer = null;


///////////////////////////////////////////////////////////////////////////////
function showBody() {
	initTooltips();
	
	document.body.style.display = "block";
	document.body.style.visibility = "visible";
	document.body.style.opacity = "1";
}

function setTheme() {
	const themeToggle = document.getElementById('themechange');
	const prefersDark = window.matchMedia('(prefers-color-scheme: dark)');

	function applyTheme(theme) {
		document.body.classList.toggle('dark-mode', theme === 'dark');
		updateThemeBackgroundImages(theme);
		document.getElementById('theme-mode-css').setAttribute(
		  'href',
		  'lib/layouts/colors/' + theme + '/' + theme + '.css'
		);
		document.getElementById('theme-default-css').setAttribute(
		  'href',
		  'lib/layouts/colors/' + theme + '/themes/Default/default.css'
		);
	}

	const savedTheme = sessionStorage.getItem('theme');

	if (savedTheme) {
		applyTheme(savedTheme);
	} else {
		applyTheme(prefersDark.matches ? 'dark' : 'light');
	}

	themeToggle.addEventListener('click', function () {
		const isDark = document.body.classList.contains('dark-mode');
		const nextTheme = isDark ? 'light' : 'dark';

		applyTheme(nextTheme);
		sessionStorage.setItem('theme', nextTheme);
	});

	prefersDark.addEventListener('change', function (e) {
		if (!sessionStorage.getItem('theme')) {
			applyTheme(e.matches ? 'dark' : 'light');
		}
	});
}

function updateThemeBackgroundImages(mode) {
  var elements = document.querySelectorAll('.theme-bg, input:not([type="submit"])');

  elements.forEach(function (el) {
    var bgimg = window.getComputedStyle(el).backgroundImage;

    if (bgimg && bgimg !== 'none') {
      bgimg = bgimg
        .replaceAll('/light/', '/' + mode + '/')
        .replaceAll('/dark/', '/' + mode + '/');

      el.style.backgroundImage = bgimg;
    }
  });
}


function initTooltips() {
	const X_OFFSET = 12;
	const Y_OFFSET = 28;
	const tooltip = document.getElementById('tooltip');

	document.querySelectorAll('#apptoolbar [title]').forEach(el => {
		const text = el.getAttribute('title');
		el.removeAttribute('title');

		el.addEventListener('mouseenter', () => {
			tooltip.textContent = text;
			tooltip.style.opacity = '1';
		});

		el.addEventListener('mousemove', e => {
			const ttRect = tooltip.getBoundingClientRect();
			const vw = window.innerWidth;
			const vh = window.innerHeight;

			let x = e.clientX + X_OFFSET;
			let y = e.clientY + Y_OFFSET;

			/* Clamp horizontally */
			if (x + ttRect.width > vw) {
				x = e.clientX - ttRect.width - X_OFFSET;
			}
			if (x < 0) {
				x = 0;
			}

			/* Clamp vertically */
			if (y + ttRect.height > vh) {
				y = e.clientY - ttRect.height - Y_OFFSET;
			}
			if (y < 0) {
				y = 0;
			}

			tooltip.style.left = x + 'px';
			tooltip.style.top = y + 'px';
		});

		el.addEventListener('mouseleave', () => {
			tooltip.style.opacity = '0';
		});
	});
}

///////////////////////////////////////////////////////////////////////////////

function updateToggleLabels(row) {
	const checkbox = row.querySelector("input[type='checkbox']");
	const leftLabel = row.querySelector(".toggle-label-left");
	const rightLabel = row.querySelector(".toggle-label-right");

	// Remove current highlights
	leftLabel?.classList.remove("selected");
	rightLabel?.classList.remove("selected");

	// Add correct one based on state
	if (checkbox.checked) {
		rightLabel?.classList.add("selected");
	} else {
		leftLabel?.classList.add("selected");
	}
}

///////////////////////////////////////////////////////////////////////////////

function convertToToggle(selector = ".auto-toggle") {
  document.querySelectorAll(selector).forEach(original => {
    if (original.dataset.converted === "yes") return;
    original.dataset.converted = "yes";

    const parent = original.parentNode;
    const leftText = original.dataset.labelLeft;
    const rightText = original.dataset.labelRight;

    // ✔ THIS is the row container
    const row = document.createElement("span");
    row.className = "toggle-container";   // or toggle-row (your naming)

    // Insert before existing checkbox
    parent.insertBefore(row, original);

    // Left label
    if (leftText) {
      const leftLabel = document.createElement("span");
      leftLabel.className = "toggle-label toggle-label-left";
      leftLabel.textContent = leftText;
      row.appendChild(leftLabel);
    }

    // Toggle wrapper
    const wrapper = document.createElement("span");
    wrapper.className = "toggle-wrapper";
    row.appendChild(wrapper);

    // Move original checkbox into wrapper
    wrapper.appendChild(original);

    // Slider
    const slider = document.createElement("span");
    slider.className = "toggle-slider";
    original.insertAdjacentElement("afterend", slider);

    // Right label
    if (rightText) {
      const rightLabel = document.createElement("span");
      rightLabel.className = "toggle-label toggle-label-right";
      rightLabel.textContent = rightText;
      row.appendChild(rightLabel);
    }

    // Listen for changes and update highlights
    original.addEventListener("change", () => updateToggleLabels(row));

    // Set initial highlight
    updateToggleLabels(row);
  });
}

	
///////////////////////////////////////////////////////////////////////////////

function initEventButtons() {
	document.querySelectorAll(".event").forEach(function(el) {
      el.addEventListener("click", function() {
        var target = el.getAttribute("data-id"); // e.g. "dashboard"
        openModule(target);
      });
    });
	
	convertToToggle();
	setTheme();
}

async function sendCsvToAddApi_JS(csvString, categoryId) {

    const form = new FormData();
    const blob = new Blob([csvString], { type: "text/csv" });

    form.append("csv_file", blob, "domains.csv");
    form.append("csv_sep", ",");
    if (categoryId) {
        form.append("cid", categoryId);
    }

    const url = API_BASE_URL + "?c=add&t=domain&oper=csv";

    const res = await fetch(url, {
        method: "POST",
        credentials: "same-origin",  // Important: sends PHP session cookie
        body: form
    });

    const text = await res.text();

    try {
        return JSON.parse(text);
    } catch (e) {
        return {
            status: "error",
            error: "Non-JSON response from API",
            raw: text
        };
    }
}

async function get_config_data(configname, API_BASE_URL) {
  // Build URL-encoded POST data
  const bodyData = new URLSearchParams({
	c: "get",
	t: "config",
	oper: "get",
	configname: `config_${configname}`
  });

  const response = await fetch(API_BASE_URL, {
	method: "POST",
	headers: {
	  "Content-Type": "application/x-www-form-urlencoded"
	},
	body: bodyData.toString()
  });

  const text = await response.text();
  const json = JSON.parse(text);

  if (json.status === "ok") {
	return json.data; // this is the raw string stored in PHP
  } else {
	throw new Error(json.error || "Failed to fetch config data");
  }
}

async function set_config_data(configname, configdata, API_BASE_URL) {
  const bodyData = new URLSearchParams({
    c: "admin",
    t: "config",
    oper: "set",
    configname: `config_${configname}`,
    configdata: JSON.stringify(configdata)
  });

  const response = await fetch(API_BASE_URL, {
    method: "POST",
    headers: {
      "Content-Type": "application/x-www-form-urlencoded"
    },
    body: bodyData.toString()
  });

  const json = await response.json();

  if (json.status === "ok") {
    return true;
  } else {
    throw new Error(json.error || "Failed to set config data");
  }
}

///////////////////////////////////////////////////////////////////////////////

function openModule(target) {
	if (target) {
		if(target == "logout")
			window.location.href = baseurl + "/?" + target;
		else if(target == "tool-add")
			window.location.href = baseurl + "/tools/add-domains.php";
		else if(target == "tool-export")
			window.location.href = baseurl + "/tools/export-domains.php";
		else if(target == "customtools")
			window.location.href = baseurl + "/tools/";
		else
			window.location.href = baseurl + "/" + target + ".php";
	}
}

///////////////////////////////////////////////////////////////////////////////

function startAuthSessionMonitor() {
	if (authSessionTimer !== null) return;
	authSessionTimer = setInterval(async () => {
		await checkAuthSession();
	}, 60 * 1000);
}

///////////////////////////////////////////////////////////////////////////////

function showLoginRequired() {
	// Stop logged-in session monitor
	if (authSessionTimer !== null) {
		clearInterval(authSessionTimer);
		authSessionTimer = null;
	}

	var notloggedin = document.getElementById('loginrequired');
	if (!notloggedin) return;
	notloggedin.style.display = 'block';

	var contentblock = document.getElementById('content');
	if (!contentblock) return;
	contentblock.style.display = 'none';

	// Start login polling if not already running
	if (loginPollTimer !== null) return;

	loginPollTimer = setInterval(async () => {
		const authstatus = await checkAuthSession();

		if (authstatus === true) {
			clearInterval(loginPollTimer);
			loginPollTimer = null;
			location.reload();
		}
	}, 60 * 1000);
}



///////////////////////////////////////////////////////////////////////////////

async function checkAuthSession() {
	// Call API with no parameters. We use validate===0 to detect login.
	const url = new URL(API_BASE_URL);
	url.searchParams.set('sw', '0');
	try {
		const res = await fetch(url.toString(), { credentials: 'same-origin' });
		const data = await res.json();

		// Must have validate === 0 to be accepted
		if (!data || typeof data.validate === 'undefined' || data.validate !== 0) {
			showLoginRequired();
			showBody();
			return false;
		}

		// Session IS valid
		return true;

	} catch (e) {
		document.body.innerHTML = `
			<div style="max-width:600px;margin:40px auto;padding:20px;
				background:#fff;border:1px solid #ccc;border-radius:6px;
				font-family:sans-serif;font-size:16px;">
				<h2>API Connection Error</h2>
				<p>Unable to contact the API. This may indicate:</p>
				<ul>
					<li>You are not logged in</li>
					<li>Session expired</li>
					<li>Incorrect API URL</li>
					<li>Network problem</li>
				</ul>
			</div>
		`;
		return false;
	}
}